'use client';

import { useEffect, useState } from 'react';
import { useRouter, useParams } from 'next/navigation';
import Link from 'next/link';
import { supabase, Blog } from '@/lib/supabase';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Switch } from '@/components/ui/switch';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { ImageUpload } from '@/components/admin/ImageUpload';
import { toast } from 'sonner';
import { ArrowLeft, Loader2, Save, Eye } from 'lucide-react';

export default function EditBlogPage() {
  const router = useRouter();
  const params = useParams();
  const id = params.id as string;
  
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [form, setForm] = useState({
    title: '',
    slug: '',
    excerpt: '',
    content: '',
    cover_image: '',
    author_name: '',
    author_avatar: '',
    published: false,
  });

  useEffect(() => {
    async function fetchBlog() {
      const { data: { session } } = await supabase.auth.getSession();
      
      if (!session) {
        router.push('/admin');
        return;
      }

      const { data: adminUser } = await supabase
        .from('admin_users')
        .select('*')
        .eq('user_id', session.user.id)
        .single();

      if (!adminUser) {
        await supabase.auth.signOut();
        router.push('/admin');
        return;
      }

      const { data, error } = await supabase
        .from('blogs')
        .select('*')
        .eq('id', id)
        .single();

      if (error || !data) {
        toast.error('Blog not found');
        router.push('/admin/blogs');
        return;
      }

      const blog = data as Blog;
      setForm({
        title: blog.title,
        slug: blog.slug,
        excerpt: blog.excerpt || '',
        content: blog.content,
        cover_image: blog.cover_image || '',
        author_name: blog.author_name,
        author_avatar: blog.author_avatar || '',
        published: blog.published,
      });
      setLoading(false);
    }
    fetchBlog();
  }, [id, router]);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setSaving(true);

    try {
      const { error } = await supabase
        .from('blogs')
        .update({
          ...form,
          published_at: form.published ? new Date().toISOString() : null,
        })
        .eq('id', id);

      if (error) throw error;

      toast.success('Blog updated successfully!');
      router.push('/admin/blogs');
    } catch (error: any) {
      toast.error(error.message || 'Failed to update blog');
      setSaving(false);
    }
  };

  if (loading) {
    return (
      <main className="min-h-screen bg-background flex items-center justify-center pt-24">
        <Loader2 className="h-8 w-8 animate-spin text-primary" />
      </main>
    );
  }

  return (
    <main className="min-h-screen bg-muted/30">
      {/* Hero Background */}
      <div className="bg-[#7D2C5A] h-32" />
      
      <div className="container mx-auto px-4 max-w-4xl py-8">
        {/* Header */}
        <div className="mb-6">
          <Link 
            href="/admin/blogs" 
            className="inline-flex items-center text-muted-foreground hover:text-foreground mb-2"
          >
            <ArrowLeft className="h-4 w-4 mr-2" />
            Back to Blogs
          </Link>
          <div className="flex items-center justify-between">
            <h1 className="text-3xl font-bold text-foreground">Edit Blog</h1>
            {form.published && (
              <Link href={`/blog/${form.slug}`} target="_blank">
                <Button variant="outline">
                  <Eye className="h-4 w-4 mr-2" />
                  View Live
                </Button>
              </Link>
            )}
          </div>
        </div>

        <form onSubmit={handleSubmit}>
          <div className="grid gap-6">
            {/* Main Content */}
            <Card>
              <CardHeader>
                <CardTitle>Blog Content</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="space-y-2">
                  <Label htmlFor="title">Title *</Label>
                  <Input
                    id="title"
                    placeholder="Enter blog title"
                    value={form.title}
                    onChange={(e) => setForm({ ...form, title: e.target.value })}
                    required
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="slug">Slug *</Label>
                  <Input
                    id="slug"
                    placeholder="blog-post-url"
                    value={form.slug}
                    onChange={(e) => setForm({ ...form, slug: e.target.value })}
                    required
                  />
                  <p className="text-sm text-muted-foreground">
                    URL: /blog/{form.slug || 'your-slug'}
                  </p>
                </div>

                <div className="space-y-2">
                  <Label htmlFor="excerpt">Excerpt</Label>
                  <Textarea
                    id="excerpt"
                    placeholder="Brief summary of the blog post..."
                    value={form.excerpt}
                    onChange={(e) => setForm({ ...form, excerpt: e.target.value })}
                    rows={3}
                  />
                </div>

                <div className="space-y-2">
                  <Label htmlFor="content">Content *</Label>
                  <Textarea
                    id="content"
                    placeholder="Write your blog content here..."
                    value={form.content}
                    onChange={(e) => setForm({ ...form, content: e.target.value })}
                    rows={15}
                    required
                    className="font-mono text-sm"
                  />
                </div>

                <ImageUpload
                  label="Cover Image"
                  value={form.cover_image}
                  onChange={(url) => setForm({ ...form, cover_image: url })}
                  folder="covers"
                  placeholder="https://example.com/image.jpg"
                />
              </CardContent>
            </Card>

            {/* Author & Settings */}
            <Card>
              <CardHeader>
                <CardTitle>Author & Settings</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="space-y-2">
                    <Label htmlFor="author_name">Author Name</Label>
                    <Input
                      id="author_name"
                      placeholder="Author Name"
                      value={form.author_name}
                      onChange={(e) => setForm({ ...form, author_name: e.target.value })}
                    />
                  </div>
                  <ImageUpload
                    label="Author Avatar"
                    value={form.author_avatar}
                    onChange={(url) => setForm({ ...form, author_avatar: url })}
                    folder="avatars"
                    placeholder="https://example.com/avatar.jpg"
                  />
                </div>

                <div className="flex items-center justify-between p-4 bg-muted rounded-lg">
                  <div>
                    <Label htmlFor="published" className="font-medium">Published</Label>
                    <p className="text-sm text-muted-foreground">
                      Make this blog visible to the public
                    </p>
                  </div>
                  <Switch
                    id="published"
                    checked={form.published}
                    onCheckedChange={(checked) => setForm({ ...form, published: checked })}
                  />
                </div>
              </CardContent>
            </Card>

            {/* Actions */}
            <div className="flex gap-3 justify-end">
              <Link href="/admin/blogs">
                <Button type="button" variant="outline">Cancel</Button>
              </Link>
              <Button type="submit" disabled={saving}>
                {saving ? (
                  <>
                    <Loader2 className="h-4 w-4 mr-2 animate-spin" />
                    Saving...
                  </>
                ) : (
                  <>
                    <Save className="h-4 w-4 mr-2" />
                    Save Changes
                  </>
                )}
              </Button>
            </div>
          </div>
        </form>
      </div>
    </main>
  );
}
